/************************************************************************************************************\

Module Name:    ModuleSync.h

Description:    .

    Copyright (c) 2016, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

#ifndef INC_ModuleSync_H
#define INC_ModuleSync_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------
#include "Liberatus.h"
#include "CommonUtils.h"

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Enumeration:      ModuleSyncStreamState

Description:    .

\************************************************************************************************************/
typedef enum
{
    StreamState_NoBuffer      = 0,  // The stream hasn't received any buffer yet.
    StreamState_BufferReceived,     // The stream began receiving buffers.
    StreamState_BufferingUp,        // The stream is building its input queue toward initial target level of
                                    // buffering.
    StreamState_BufferingDown,      // The stream has exceeded its initial target level of buffering and is
                                    // dropping buffers to reach it.
    StreamState_Buffered,           // The stream has reached its initial target level of buffering. It is
                                    // waiting for the other stream to do so, dropping buffers to keep its
                                    // level.
    StreamState_Aligning,           // The stream is dropping old buffers to align its base times with the
                                    // other stream.
    StreamState_ReadyToRender       // The queue is buffered and ready to start rendering buffers at the sink.

} ModuleSyncStreamState;

/************************************************************************************************************\

Structure:      ModuleSyncStreamInfo

Description:    .

\************************************************************************************************************/
typedef struct
{
    MBOOL   bIsConfigured;
    MUINT64 uiBaseTime;             // First buffer timestamp un microsecond.
    MUINT64 uiLastBufTsUsec;        // Last updated buffer timestamp in microseconds.
    MUINT64 uiLastValidBufTsUsec;   // Last valid updated buffer timestamp in microseconds. (valid when larger
                                    // than previous one).
    MUINT64 uiLastBufSysTimeUsec;   // Snapshot of the system time when the last valid timestamp was updated.
    MUINT   uiCurQueueSize;         // Current number of buffers in the input queue.
    MBOOL   bBaseTimeSet;           // Set once the stream has set the base time or updated the first timestamp.
    MBOOL   bFirstTsUpdated;        // Set once the stream has updated the first timestamp.
    MUINT64 uiBufferTimeUsec;       // Buffer time in microseconds.
    MUINT   uiInputQueueLength;     // Maximum number of buffer that can be stored in the input queue of the
                                    // sink.
    MUINT   uiTargetQueueSize;      // Target number of buffers stored in the queue at the beginning of the
                                    // rendering.
    MUINT64 uiTargetBaseTime;       // Target base time that the stream with the highest latency must try to
                                    // reach. This is the base time of the stream with the lowest latency at
                                    // the moment the other stream receives its first buffer.
    ModuleSyncStreamState eState;
    const char* szName;

} ModuleSyncStreamInfo;

/************************************************************************************************************\

Structure:      ModuleSyncMaster

Description:    .

\************************************************************************************************************/
typedef struct
{
    ModuleSyncStreamInfo    oSlvInfo;
    ModuleSyncStreamInfo    oMstInfo;
    LMutext_Handle          hMutex;

} ModuleSyncMaster;

/************************************************************************************************************\

Structure:      ModuleSync

Description:    All times are in nanosecond (1/1000000000 second). Duration precision is important because
                timestamp and present time will be calculated incrementally.

\************************************************************************************************************/
typedef struct tagModuleSync
{
    MBOOL           bIsConfigured;
    MINT64          iTimestampOffset;           // Offset between timestamp and present time.
    MUINT64         uiInFrameDuration;          // Input frame duration.
    MUINT64         uiOutFrameDuration;         // Output frame duration.
    MUINT64         uiTimestamp;                // Timestamp expected for the next frame.
    MUINT64         uiMaxTimestampDrift;        // Max. diff. allowed between the actual and the expected
                                                // timestamp.
    MUINT64         uiPresentTime;              // Next time where a frame can be presented.
    MUINT64         uiMaxPresentTimeDrift;      // Max. diff. allowed between the current and the present time.
    MUINT           uiMaxQueueLengthAllowed;    // Used to control queue overflowing.
    MBOOL           bLowLatency;                // Keep latency as low as possible.
    MUINT           uiLowlOverQueueCount;       // Number of frames played in low latency with a queue length
                                                // greater than 0.
    MBOOL           bResync;                    // Resynchronize timestamp and present time with system time.
    char            szQueueLevel[32];           // Used for debugging only.

    // For synchronization with external master stream.
    MBOOL           bExternalMasterSync;
    MUINT64         uiMaxMstStrmDriftAllowed;   // Maximum drift from the master stream allowed before resync.
    ModuleSyncMaster oModSyncMst;               // Master stream synchronization data.
    MUINT64         uiPrevTimestampNsec;        // Previous timestamp in nanoseconds.

} ModuleSync;

#define ModuleSyncStreamInfo_Construct { \
                    /*.bIsConfigured                =*/ MFALSE, \
                    /*.uiBaseTime                   =*/ 0, \
                    /*.uiLastValidBufTsUsec         =*/ 0, \
                    /*.uiLastBufTsUsec              =*/ 0, \
                    /*.uiLastBufSysTimeUsec         =*/ 0, \
                    /*.uiCurQueueSize               =*/ 0, \
                    /*.bBaseTimeSet                 =*/ MFALSE, \
                    /*.bFirstTsUpdated              =*/ MFALSE, \
                    /*.uiBufferTimeUsec             =*/ 0, \
                    /*.uiInputQueueLength           =*/ 0, \
                    /*.uiTargetQueueSize            =*/ 0, \
                    /*.uiTargetBaseTime             =*/ 0, \
                    /*.eState                       =*/ StreamState_NoBuffer, \
                    /*.szName                       =*/ "" }

#define ModuleSyncMaster_Construct { \
                    /*.oSlvInfo                     =*/ ModuleSyncStreamInfo_Construct, \
                    /*.oMstInfo                     =*/ ModuleSyncStreamInfo_Construct, \
                    /*.hMutex                       =*/ MNULL }

#define ModuleSync_Construct { \
                    /*.bIsConfigured                =*/ MFALSE, \
                    /*.iTimestampOffset             =*/ 0, \
                    /*.uiInFrameDuration            =*/ 0, \
                    /*.uiOutFrameDuration           =*/ 0, \
                    /*.uiTimestamp                  =*/ 0, \
                    /*.uiMaxTimestampDrift          =*/ 0, \
                    /*.uiPresentTime                =*/ 0, \
                    /*.uiMaxPresentTimeDrift        =*/ 0, \
                    /*.uiMaxQueueLengthAllowed      =*/ 0, \
                    /*.bLowLatency                  =*/ MFALSE, \
                    /*.uiLowlOverQueueCount         =*/ 0, \
                    /*.bResync                      =*/ MFALSE, \
                    /*.szQueueLevel                 =*/ {0}, \
                    /*.bExternalMasterSync          =*/ MFALSE, \
                    /*.uiMaxMstStrmDriftAllowed     =*/ 0, \
                    /*.oModSyncMst                  =*/ ModuleSyncMaster_Construct, \
                    /*.uiPrevTimestampNsec          =*/ 0 }

// -----------------------------------------------------------------------------------------------------------
//                  G L O B A L   V A R I A B L E / F U N C T I O N   R E F E R E N C E S
// -----------------------------------------------------------------------------------------------------------

LStatus ModSync_Init(
    ModuleSync*         poSyncData);

void ModSync_Cleanup(
    ModuleSync*         poModSync);

void ModSync_SetParams(
    ModuleSync*         poSyncData,
    MUINT64             uiInFrameDurationNsec,
    MUINT64             uiOutFrameDurationNsec,
    MUINT               uiMaxQueueLength,
    MBOOL               bControlQueueOverflow,
    MBOOL               bExternalMasterSync,
    MBOOL               bLowLatency);

MBOOL ModSync_PrePresent(
    ModuleSync*         poModSync,
    MUINT64             uiTimestampNsec,
    MUINT               uiQueueLength,
    MINT64*             piCurDriftFromMstNsec);

void ModSync_PostPresent(
    ModuleSync*         poModSync,
    MUINT64             uiTimestampNsec);

void ModSyncMst_SetParams(
    ModuleSyncMaster*   poModSyncMst,
    MUINT64             uiBufferTimeUsec,
    MUINT               uiInputQueueLength,
    MUINT               uiTargetQueueSize,
    MBOOL               bMasterStream);

void ModSyncMst_SetBaseTime(
    ModuleSyncMaster*   poModSyncMst,
    MUINT64             uiBaseTimeUsec,
    MBOOL               bMasterStream);

void ModSyncMst_UpdateTimestamp(
    ModuleSyncMaster*   poModSyncMst,
    MUINT64             uiTimestamp,
    MBOOL               bMasterStream);

void ModSyncMst_PreSync(
    ModuleSyncMaster*   poModSyncMst,
    MUINT               uiQueueSize,
    MUINT64             uiTimestampUsec,
    MBOOL               bMasterStream,
    MBOOL*              pbCanRender,
    MBOOL*              pbDropOldest);

void ModSyncMst_Lock(
    ModuleSyncMaster*   poModSyncMst);

void ModSyncMst_Unlock(
    ModuleSyncMaster*   poModSyncMst);

// -----------------------------------------------------------------------------------------------------------
//                                 I N L I N E S   D E F I N I T I O N S
// -----------------------------------------------------------------------------------------------------------

#endif // INC_ModuleSync_H
